#include "caffe2/operators/cosh_op.h"

#include <algorithm>
#include <functional>

namespace caffe2 {

template <>
template <typename T>
bool CoshGradientFunctor<CPUContext>::Forward(
    const std::vector<int>& /* dY_dims */,
    const std::vector<int>& X_dims,
    const T* dY,
    const T* X,
    T* dX,
    CPUContext* /* context */) const {
  const int size = std::accumulate(
      X_dims.cbegin(), X_dims.cend(), 1, std::multiplies<int>());
  ConstEigenVectorArrayMap<T> dY_arr(dY, size);
  ConstEigenVectorArrayMap<T> X_arr(X, size);
  EigenVectorMap<T>(dX, size) = dY_arr * (X_arr.exp() - (-X_arr).exp()) / 2;
  return true;
}

REGISTER_CPU_OPERATOR(
    Cosh,
    UnaryElementwiseOp<
        TensorTypes<float>,
        CPUContext,
        CoshFunctor<CPUContext>>);
REGISTER_CPU_OPERATOR(
    CoshGradient,
    BinaryElementwiseOp<
        TensorTypes<float>,
        CPUContext,
        CoshGradientFunctor<CPUContext>>);

OPERATOR_SCHEMA(Cosh)
    .NumInputs(1)
    .NumOutputs(1)
    .IdenticalTypeAndShape()
    .SetDoc(R"DOC(
Calculates the hyperbolic cosine of the given input tensor, element-wise.

Github Links:

- https://github.com/pytorch/pytorch/blob/master/caffe2/operators/cosh_op.cc


<details>

<summary> <b>Example</b> </summary>

**Code**

```

workspace.ResetWorkspace()

op = core.CreateOperator(
    "Cosh",
    ["X"],
    ["Y"]
)

workspace.FeedBlob("X", np.random.rand(5).astype(np.float32))
print("X:", workspace.FetchBlob("X"))
workspace.RunOperatorOnce(op)
print("Y:", workspace.FetchBlob("Y"))

```

**Result**

```

X: [0.66423494 0.32074615 0.81523746 0.90423071 0.39275789]
Y: [1.22883528 1.05188156 1.35112322 1.43744212 1.07812598]

```

</details>

)DOC")
    .Input(0, "input", "Input tensor")
    .Output(
        0,
        "output",
        "The hyperbolic cosine values of the input tensor, computed "
        "element-wise")
    .InheritOnnxSchema();

OPERATOR_SCHEMA(CoshGradient)
    .NumInputs(2)
    .NumOutputs(1)
    .IdenticalTypeAndShapeOfInput(0);

namespace {

class GetCoshGradient : public GradientMakerBase {
  using GradientMakerBase::GradientMakerBase;
  std::vector<OperatorDef> GetGradientDefs() override {
    return SingleGradientDef(
        "CoshGradient",
        "",
        std::vector<std::string>{GO(0), I(0)},
        std::vector<std::string>{GI(0)});
  }
};

} // namespace

REGISTER_GRADIENT(Cosh, GetCoshGradient);

} // namespace caffe2
